document.addEventListener('DOMContentLoaded', () => {
    const apiDocsContainer = document.getElementById('api-docs');

    // URL base real del servicio de facturación (producción)
    const BASE_SOAP_URL = 'https://cucc.com.mx/apifacturacion/public/soap/facturas';

    // --- Definición de las operaciones de la API ---

    const apiOperations = [
        {
            name: 'TimbrarFacturaTXT',
            summary: 'Genera un CFDI 4.0 a partir de un archivo de texto formateado (texto -> XML -> timbrado).',
            method: 'POST',
            path: '/apifacturacion/public/soap/facturas',
            parameters: [
                { name: 'usuario', type: 'string', description: 'Usuario de autenticación (RFC emisor).' },
                { name: 'pass', type: 'string', description: 'Contraseña del usuario/autenticación.' },
                { name: 'file', type: 'string', description: 'Contenido del archivo TXT en Base64.' }
            ],
            soapRequestTemplate: `
<?xml version="1.0" encoding="utf-8"?>
<soapenv:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                  xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                  xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:urn="urn:cuccws">
    <soapenv:Header/>
    <soapenv:Body>
        <urn:invoiceTxt soapenv:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/">
            <requestinvoicetxt xsi:type="urn:requestinvoicetxt">
                <usuario xsi:type="xsd:string">{{usuario}}</usuario>
                <pass xsi:type="xsd:string">{{pass}}</pass>
                <file xsi:type="xsd:string">{{file}}</file>
            </requestinvoicetxt>
        </urn:invoiceTxt>
    </soapenv:Body>
</soapenv:Envelope>`.trim()
        },

        // ✅ Timbrar XML usando doInvoice + requestinvoice + parámetro file
        {
            name: 'TimbrarFacturaXML',
            summary: 'Genera un CFDI 4.0 a partir de un archivo XML en Base64 (CFDI sin timbrar).',
            method: 'POST',
            path: '/apifacturacion/public/soap/facturas',
            parameters: [
                { name: 'usuario', type: 'string', description: 'Usuario de autenticación (RFC emisor).' },
                { name: 'pass', type: 'string', description: 'Contraseña del usuario/autenticación.' },
                { name: 'file', type: 'string', description: 'Archivo XML CFDI en Base64 (sin timbrar).' }
            ],
            soapRequestTemplate: `
<?xml version="1.0" encoding="utf-8"?>
<soapenv:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                  xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                  xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:urn="urn:cuccws">
    <soapenv:Header/>
    <soapenv:Body>
        <urn:doInvoice soapenv:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/">
            <requestinvoice xsi:type="urn:requestinvoice">
                <usuario xsi:type="xsd:string">{{usuario}}</usuario>
                <pass xsi:type="xsd:string">{{pass}}</pass>
                <file xsi:type="xsd:string">{{file}}</file>
            </requestinvoice>
        </urn:doInvoice>
    </soapenv:Body>
</soapenv:Envelope>`.trim()
        },

        // ✅ CORREGIDO: CancelarFactura basado en el cURL que enviaste
        {
            name: 'CancelarFactura',
            summary: 'Cancela un CFDI previamente timbrado.',
            method: 'POST',
            path: '/apifacturacion/public/soap/facturas',
            parameters: [
                { name: 'usuario', type: 'string', description: 'Usuario de autenticación (RFC emisor).' },
                { name: 'pass', type: 'string', description: 'Contraseña del usuario/autenticación.' },
                { name: 'folio', type: 'string', description: 'Folio de la factura a cancelar (ej. X100).' },
                { name: 'razon', type: 'string', description: 'Motivo SAT de cancelación (01, 02, 03, 04).' }
            ],
            soapRequestTemplate: `
<?xml version="1.0" encoding="utf-8"?>
<soapenv:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                  xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                  xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:urn="urn:cuccws">
    <soapenv:Header/>
    <soapenv:Body>
        <urn:cancelInvoice soapenv:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/">
            <requestCancelInvoice xsi:type="urn:requestCancelInvoice">
                <usuario xsi:type="xsd:string">{{usuario}}</usuario>
                <pass xsi:type="xsd:string">{{pass}}</pass>
                <folio xsi:type="xsd:string">{{folio}}</folio>
                <razon xsi:type="xsd:string">{{razon}}</razon>
            </requestCancelInvoice>
        </urn:cancelInvoice>
    </soapenv:Body>
</soapenv:Envelope>`.trim()
        }
    ];
    // --- Fin de la definición ---

    // Renderizar todas las operaciones de la API
    apiOperations.forEach(op => {
        const opElement = createOperationElement(op);
        apiDocsContainer.appendChild(opElement);
    });

    // Abrir la primera operación por defecto
    const firstOpContent = apiDocsContainer.querySelector('.op-content');
    if (firstOpContent) {
        firstOpContent.classList.add('active');
    }

    // Crea la tarjeta HTML para cada operación
    function createOperationElement(op) {
        const operationDiv = document.createElement('div');
        operationDiv.className = 'operation';

        const soapBlockId = `soap-template-${op.name}`;
        const headersBlockId = `headers-template-${op.name}`;

        operationDiv.innerHTML = `
            <div class="op-header" data-op-name="${op.name}">
                <h2>
                    <span class="op-method post">${op.method}</span>
                    <span class="op-path">${op.name}</span>
                </h2>
                <span class="op-summary">${op.summary}</span>
            </div>

            <div class="op-content" id="content-${op.name}">
                <h3>Parámetros</h3>
                <div class="parameters">
                    <table class="param-table">
                        <thead>
                            <tr>
                                <th>Nombre</th>
                                <th>Tipo</th>
                                <th>Descripción</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${op.parameters.map(p =>
                                `<tr>
                                    <td>${p.name}</td>
                                    <td>${p.type}</td>
                                    <td>${p.description}</td>
                                </tr>`).join('')}
                        </tbody>
                    </table>
                </div>

                <div class="try-out">
                    <h3>Pruébalo (Try it out)</h3>

                    <div class="form-group">
                        <label for="api-endpoint-url-${op.name}">Endpoint URL</label>
                        <input
                            type="text"
                            id="api-endpoint-url-${op.name}"
                            value="${BASE_SOAP_URL}"
                            placeholder="URL del Endpoint SOAP">
                    </div>

                    ${op.parameters.map(p => `
                        <div class="form-group">
                            <label for="param-${op.name}-${p.name}">${p.name}</label>
                            <input
                                type="text"
                                id="param-${op.name}-${p.name}"
                                placeholder="${p.description}">
                        </div>
                    `).join('')}

                    <button class="btn btn-execute" data-op-name="${op.name}">
                        Ejecutar
                    </button>
                </div>

                <div class="tech-details">
                    <h3>Estructura SOAP (request)</h3>
                    <p>
                        Ejemplo de sobre SOAP enviado al endpoint para <strong>${op.name}</strong>.
                        Sustituye los campos entre <code>{{'{'}}llaves{{'}'}}</code> con tus valores.
                    </p>
                    <pre class="code-block" id="${soapBlockId}"></pre>

                    <h3>Headers HTTP</h3>
                    <p>Headers mínimos recomendados para consumir este método:</p>
                    <pre class="code-block" id="${headersBlockId}"></pre>

                    ${op.name === 'TimbrarFacturaTXT' ? `
                        <div class="form-group">
                            <a href="examples/FacturaTXT.txt" download="FacturaTXT.txt" class="btn btn-download">
                                Descargar ejemplo de TXT
                            </a>
                        </div>
                    ` : ''}

                    ${op.name === 'TimbrarFacturaXML' ? `
                        <div class="form-group">
                            <a href="examples/FacturaXML.xml" download="FacturaXML.xml" class="btn btn-download">
                                Descargar ejemplo de XML
                            </a>
                        </div>
                    ` : ''}

                    ${(op.name === 'TimbrarFacturaTXT' || op.name === 'TimbrarFacturaXML') ? `
                        <div class="response-explanation">
                            <h4>Formato de Respuesta</h4>
                            <p>
                                En caso de éxito, la API regresará un objeto SOAP que contendrá el PDF y el XML de la factura timbrada, ambos en formato Base64.
                            </p>
                            <p>
                                En caso de error, la API regresará un mensaje indicando el problema específico, que podrás ver en la sección "Cuerpo de la Respuesta".
                            </p>
                        </div>
                    ` : ''}
                </div>

                <div class="response" id="response-${op.name}" style="display:none;">
                    <h3>Respuesta del Servidor</h3>
                    <h4>Cuerpo de la Respuesta</h4>
                    <pre class="code-block" id="response-body-${op.name}"></pre>
                    <h4>Cabeceras</h4>
                    <pre class="code-block" id="response-headers-${op.name}"></pre>
                </div>
            </div>
        `;

        // Rellenar el bloque de plantilla SOAP con texto plano
        const soapPre = operationDiv.querySelector(`#${soapBlockId}`);
        if (soapPre) {
            soapPre.textContent = op.soapRequestTemplate;
        }

        // Rellenar el bloque de headers con un ejemplo tipo HTTP/cURL
        const headersPre = operationDiv.querySelector(`#${headersBlockId}`);
        if (headersPre) {
            const headersText = [
                `POST ${BASE_SOAP_URL} HTTP/1.1`,
                `Host: cucc.com.mx`,
                `Content-Type: text/xml`,
                ``,
                `// Ejemplo cURL equivalente:`,
                `curl --location '${BASE_SOAP_URL}' \\`,
                `  --header 'Content-Type: text/xml' \\`,
                `  --data '<soapenv:Envelope>...</soapenv:Envelope>'`
            ].join('\n');
            headersPre.textContent = headersText;
        }

        return operationDiv;
    }

    // Expandir/colapsar la operación
    apiDocsContainer.addEventListener('click', (e) => {
        const header = e.target.closest('.op-header');
        if (header) {
            const opName = header.dataset.opName;
            const content = document.getElementById(`content-${opName}`);
            if (content) content.classList.toggle('active');
        }
    });

    // Manejar click en "Ejecutar"
    apiDocsContainer.addEventListener('click', async (e) => {
        if (!e.target.classList.contains('btn-execute')) return;

        const opName = e.target.dataset.opName;
        const operation = apiOperations.find(op => op.name === opName);
        if (!operation) return;

        let soapRequest = operation.soapRequestTemplate;

        // Sustituir parámetros en el template
        operation.parameters.forEach(p => {
            const input = document.getElementById(`param-${opName}-${p.name}`);
            const value = input ? input.value : '';
            const safeValue = escapeXml(value);
            const regex = new RegExp(`{{${p.name}}}`, 'g');
            soapRequest = soapRequest.replace(regex, safeValue);
        });

        const endpointUrlInput = document.getElementById(`api-endpoint-url-${opName}`);
        const endpointUrl = endpointUrlInput ? endpointUrlInput.value : BASE_SOAP_URL;

        const responseContainer = document.getElementById(`response-${opName}`);
        const responseBodyEl = document.getElementById(`response-body-${opName}`);
        const responseHeadersEl = document.getElementById(`response-headers-${opName}`);

        responseBodyEl.textContent = 'Ejecutando petición SOAP...';
        responseHeadersEl.textContent = '';
        responseContainer.style.display = 'block';

        try {
            const response = await fetch(endpointUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'text/xml'
                },
                body: soapRequest
            });

            // Cabeceras
            const headers = {};
            for (const [key, value] of response.headers.entries()) {
                headers[key] = value;
            }
            responseHeadersEl.textContent = JSON.stringify(headers, null, 2);

            // Cuerpo
            const responseText = await response.text();
            responseBodyEl.textContent = responseText;

        } catch (error) {
            responseBodyEl.textContent = `Error en la solicitud: ${error.message}`;
            console.error('Error al ejecutar la operación SOAP:', error);
        }
    });

    // Escapa caracteres especiales de XML para evitar romper el sobre SOAP
    function escapeXml(str) {
        if (str == null) return '';
        return String(str).replace(/[<>&'"]/g, function (c) {
            switch (c) {
                case '<': return '&lt;';
                case '>': return '&gt;';
                case '&': return '&amp;';
                case '\'': return '&apos;';
                case '"': return '&quot;';
                default: return c;
            }
        });
    }
});
